package tno.geoenergy.doubletcalc;

/**
 * @author Van Hooff, Kronimus
 * 
 * 
 *         (c) 2009 TNO
 *         
 *   Disclaimer
 *	 see
 *   http://www.tno.nl/downloads/Disclaimer%20websites%20TNO.UK1.pdf
 *	 and
 *	 http://www.nlog.nl/nl/home/termsNLOG.html         
 * 
 */

public class TnoPipe {

	// The pipe is for calculation purposes divided in segments
	// Between two segments is a pipe node
	// The number of segments is SegmentsCounts
	// The number of nodes is NodesCount = SegmentsCount+1
	// Usually, when dealing with a segment:
	// iNodePrev is the entry node of the segment (from a calculation
	// perspective)
	// iNode is the exit node of the segment (from a calculation perspective)
	// Pipe "starts/begins" at Node=0 and "ends" at Node=NodesCount-1
	// "PipeUp" is from begin to end; "PipeDown" from end to begin
	// Positive flow is from begin to end; negative flow is from end to begin
	// dL, the length along pipe difference between two nodes
	// In calculations, dL>0 indicates that iNode is more downstream/PipeUp than
	// iNodePrev, and visa versa

	// double pumpZ;
	public double[] lNode; // Length Along Pipe of Node
	public double[] zNode; // z-coordinate of Node
	public double[] pressureNode; // (Pa)
	public double[] temperatureNode; // (degC)
	public double[] tempEnvironmentNode; // (degC)
	public double[] salinityNode; // (1)
	public double[] densityNode; // (kg/m3)
	public double[] viscosityNode; // (Pa.s)
	public double[] heatCapacityNode; // (J/(kg.K))
	public double[] qVolNode; // (m3/s)
	public double[] dPgravSegment; // (Pa)
	public double[] dPviscSegment; // (Pa)
	public double[] dPpumpSegment; // (Pa)

	public boolean notConverting = false;

	// pipe specification properties
	public String Name;
	public double Length; // (m)
	public double[] InnerDiameter; // (m)
	public double OuterDiameter; // (m)
	public double[] Roughness; // (m)
	public double Zbegin; // (m)
	public double Zend; // (m)

	public double[] SectionDepthAH;
        public double[] SectionDepthTVD;
        public double[] SectionAngle;

	public double[] InnerDiameterSegment; //(m) 
	public double[] RoughnessSegment; //(m)
	public double[] SegmentAngle;
	public double[] SegmentIndex;

	// pump specification properties
	private double PumpZ; // (m); can only be set through SetPump()
	public double PumpPressure; // (Pa)
	public double PumpMaxRate; // (m3/s)
	public double PumpEfficiency; // (1)

	// temperature calculation option
	public PipeTempCalcMethod TemperatureCalculationOption;

	// results last calculate case
	public double Qmass; // (kg/s)
	public double QvolAvg;// (m3/s); average volumetric rate
	public double DpTotal;// (Pa); p[end] - p[begin]
	public double DTTotal;// (degC) T[end] - T[begin]
	public double DpGravTotal;// (Pa); pGav[end] - pGav[begin]
	public double DpViscTotal;// (Pa.s); pVisc[end] - pVisc[begin]
	public double DpPumpTotal;// (Pa); pPump[end] - pPump[begin]

	// mainly used internally
	public int SegmentsCount;
	public double SegmentLength; // (m)
	public int NodesCount;
	public int PumpSegment; // number of Segment with Pump
	public boolean PumpPresent;
	
        public TnoPipe(){
            
        }

	public TnoPipe(String name, double pipeScaling, double[] depthSpec,
			double[] segmentSpec, double diamOutSpec, double[] diamInSpec,
			double[] roughSpec, double segmentLengthTarget, boolean pumpPresent,
			double pumpZSpec, double pumpPressureSpec, double pumpMaxRateSpec,
			double pumpEfficiencySpec) {
		Name = name;
                SectionDepthAH = segmentSpec.clone();
                SectionDepthTVD = depthSpec.clone();
                SectionAngle = new double[segmentSpec.length];

                //calc tubing section average angle
		double md_prev = 0;
		double tvd_prev = 0;

		for (int s=0; s < segmentSpec.length; s++){
			SectionAngle[s] = Math.acos((SectionDepthTVD[s]-tvd_prev)/(SectionDepthAH[s]-md_prev));
			tvd_prev = SectionDepthTVD[s];
			md_prev = SectionDepthAH[s];
		}
                //adjust well length compared to the basecase scenario.
                //select thickest well segment
                int segmentToUpdate = 0;
                double diameter = 0;
                for (int s=0; s < diamInSpec.length; s++){
                    if (diamInSpec[s] > diameter){
                        diameter = diamInSpec[s];
                        segmentToUpdate = s;
                    }
                }
                //Strech or shrink the section, and subsequenty move
                double dTVD;
                if (segmentToUpdate == 0){
                    // in case segment verry short
                    if (SectionDepthTVD[segmentToUpdate] < pipeScaling){
                        dTVD = SectionDepthTVD[segmentToUpdate];
                    }else{
                        dTVD = pipeScaling;
                    }
                }
                else{
                   // in case segment verry short
                   if (SectionDepthTVD[segmentToUpdate] - SectionDepthTVD[segmentToUpdate-1] < pipeScaling){
                       dTVD = SectionDepthTVD[segmentToUpdate] - SectionDepthTVD[segmentToUpdate-1];
                   }else{
                       dTVD = pipeScaling;
                   }
                }
                double dAH = dTVD / Math.cos(SectionAngle[segmentToUpdate]);
               
                for (int s=segmentToUpdate; s < segmentSpec.length; s++){
                    SectionDepthAH[s] = SectionDepthAH[s] + dAH;
                    SectionDepthTVD[s] = SectionDepthTVD[s] + dTVD;
                }

		InnerDiameter = diamInSpec;
		OuterDiameter = diamOutSpec;
		Roughness = roughSpec;
                Length = SectionDepthAH[SectionDepthAH.length-1];

		SegmentsCount = (int) Math.round(Length / segmentLengthTarget);
		SegmentLength = Length / SegmentsCount;
		NodesCount = SegmentsCount + 1;

		InnerDiameterSegment = new double[SegmentsCount];
		RoughnessSegment = new double[SegmentsCount];
		
		lNode = new double[NodesCount];
		zNode = new double[NodesCount];
		pressureNode = new double[NodesCount];
		temperatureNode = new double[NodesCount];
		tempEnvironmentNode = new double[NodesCount];
		salinityNode = new double[NodesCount];
		densityNode = new double[NodesCount];
		viscosityNode = new double[NodesCount];
		heatCapacityNode = new double[NodesCount];
		qVolNode = new double[NodesCount];
		dPgravSegment = new double[SegmentsCount];
		dPviscSegment = new double[SegmentsCount];
		dPpumpSegment = new double[SegmentsCount];
		
		SegmentAngle = new double[SegmentsCount];
		SegmentIndex = new double[SegmentsCount];       

		//define tubing properties: innerdiameter, roughness and z
		for (int iN = 0; iN < NodesCount; iN++) {
			lNode[iN] = iN * SegmentLength;
			double[] sectionIndex = getSectionIndex(lNode[iN]);

			if (iN>0){
				zNode[iN] = zNode[iN-1] - SegmentLength * Math.cos(getSectionValue(SectionAngle, sectionIndex)); //z is negatief
                                int iS = iN-1; // first node has no segment
                                SegmentAngle[iS]= getSectionValue(SectionAngle, sectionIndex);
				InnerDiameterSegment[iS]= getSectionValue(InnerDiameter, sectionIndex);
				RoughnessSegment[iS]= getSectionValue(Roughness, sectionIndex);
				SegmentIndex[iS]= sectionIndex[0]+1;
                        }else {
                            zNode[iN] = 0; 
			}

		}
		
		//insert pump
		if (pumpPresent) {
			SetPump(pumpZSpec);
			PumpPressure = pumpPressureSpec;
			PumpMaxRate = pumpMaxRateSpec;
			PumpEfficiency = pumpEfficiencySpec;
		}

		
	}

        private double getSectionValue (double[] SectionArray, double[] sectionIndex){
            int section = (int) sectionIndex[0];
            // set to max 1
            double scale = sectionIndex[1];
            double res = 0;
            // workaround for rounding issues
            if (scale > 1){
                 scale = 1;
            }
            if (scale <1){
                if (section == SectionArray.length-1){
                     res = SectionArray[section];
                }else{
                res = scale*SectionArray[section] + (1-scale)*SectionArray[section+1];
                }
                
            }else{
                res = SectionArray[section];
            }
            return res;
        }

	private double[] getSectionIndex(double lengthAH){
                double[] res= new double[2];
		int section = -1;
                double scale = 1;
		for (int i = 0; i < SectionDepthAH.length; i++) {
			if (SectionDepthAH[i] >= lengthAH){
				section = i;
				scale = 1;
                                res[0] = section;
                                res[1] = scale;
                                return res;
			}else if (SectionDepthAH[i] >= lengthAH- SegmentLength){
				section = i;
                                scale = 1 -((lengthAH - SectionDepthAH[i])/SegmentLength);
                                res[0] = section;
                                res[1] = scale;
                                return res;
			}
		}
		return res;
	}

	public void CalcPressuresAlongPipe(double qMass, double presStartCalc,
			double tempStartCalc, double pumpPressure,
			PipeCalcDirection pipeCalcDirection,
			PipeTempCalcMethod tempCalcOption) {
		final double pbTolerance = 10.0; // tolerance on pressure balance for
											// Newton Raphson (Pa)
		final double ebTolerance = 10.0; // tolerance on energy balance for
											// Newton Raphson (W/m)

		// update properties of pipe
		PumpPressure = PumpPresent ? pumpPressure : 0;
		Qmass = qMass;

		double presk, preskm1, preskm2; // pressure @ N-R iterations k, k-1, and
										// k-2
		double tempk, tempkm1, tempkm2; // temoerature @ N-R iterations k, k-1,
										// and k-2
		double pbkm1, pbkm2; // pressure balance @ N-R iterations k-1, and k-2
		double ebkm1, ebkm2; // energy balance @ N-R iterations k-1, and k-2
		double dPBdp; // dPBdp/dp: derivative of pressure balance w.r.t.
						// pressure
		double dEBdT; // dEBdp/dT: derivate of energy balance w.r.t. temperature
		double presNodePrev, tempNodePrev, densNodePrev, viscNodePrev, cpNodePrev; // data
																					// at
																					// previous
																					// Node
		double cpNode; // heat capacity
		double densAvg, viscAvg;
		double tempEnvironment;
		double dL, dZ;
		double outerDiamSeg;

		int signFlow = 0;
		if (qMass > 0)
			signFlow = +1; // flow from beginning (iNode=0) to end of pipe
		if (qMass < 0)
			signFlow = -1; // from end to beginning (iNode=0) of pipe
		int signCalc = pipeCalcDirection == PipeCalcDirection.BeginToEnd ? +1
				: -1;

		// calculation is done segment by segment
		// for each segment with nodes iN and iN-1:
		// p[iN] = p[iN-1] + dpGravity + dpViscous + dpPump
		// dPgravity = f(dZ, density)
		// dPviscous = f(dL, Qm, density, viscosity)
		// density = average density = 0.5*(density[iN] + density[iN-1])
		// viscoisty = average viscosity = 0.5(viscoisty[iN] + viscosity[iN-1]
		// p[iN-1] is known from calculation previous segment
		// rewrite first equation:
		// p[iN] - p[iN-1] - dpGravity - dpViscous - dpPump = 0 =
		// PressureBalance = pb
		// with p[i] = p
		// func(p) = p - p[iN-1] - dpGravity - dpViscous - dpPump = 0
		// where dpGravity and dpViscous are function of p
		// to solve for func(p) = 0, use Newton Raphson:
		// p[k] = p[k-1] - func(p[k-1])/func'(p[k-1])
		// where func'(p[k-1]) = (func(p[k-1]) - func(p[k-2])/(p[k-1]) - p[k-2])

		int iNstart = pipeCalcDirection == PipeCalcDirection.BeginToEnd ? 0
				: NodesCount - 1;
		pressureNode[iNstart] = presStartCalc;
		if (tempCalcOption == PipeTempCalcMethod.Implicit)
			temperatureNode[iNstart] = tempStartCalc;

		densityNode[iNstart] = TnoWaterProperties.Density(
				pressureNode[iNstart], temperatureNode[iNstart],
				salinityNode[iNstart]);
		viscosityNode[iNstart] = TnoWaterProperties.Viscosity(
				pressureNode[iNstart], temperatureNode[iNstart],
				salinityNode[iNstart]);
		heatCapacityNode[iNstart] = TnoWaterProperties.HeatCapacity(
				pressureNode[iNstart], temperatureNode[iNstart],
				salinityNode[iNstart]);

		qVolNode[iNstart] = Qmass / densityNode[iNstart];

		for (int i = 1; i < NodesCount; i++) {
			int iN, iNprev, iSegment;

			if (pipeCalcDirection == PipeCalcDirection.BeginToEnd) {
				iN = i;
				iNprev = iN - 1;
				iSegment = iN - 1;
			} else {
				iN = NodesCount - 1 - i;
				iNprev = iN + 1;
				iSegment = iN;
			}

			dL = lNode[iN] - lNode[iNprev];
			dZ = zNode[iN] - zNode[iNprev];
			
			// outer diameter set as innerdiameter + 1 inch
			outerDiamSeg = InnerDiameterSegment[iSegment] + 1*TnoUnits.Inch_SI;

			presNodePrev = pressureNode[iNprev];
			tempNodePrev = temperatureNode[iNprev];
			densNodePrev = densityNode[iNprev];
			viscNodePrev = viscosityNode[iNprev];
			cpNodePrev = heatCapacityNode[iNprev];

			preskm1 = presNodePrev;
			densAvg = densNodePrev;
			viscAvg = viscNodePrev;
			pbkm1 = CalcPressureBalance(preskm1, presNodePrev, qMass, dZ, dL,
					densAvg, viscAvg, iSegment, signFlow * signCalc);
			presk = preskm1 - pbkm1; // smart estimate of first estimate,
										// knowing pressure balance

			if (tempCalcOption == PipeTempCalcMethod.Implicit) {
				// tempEnvironment is constant during N-R
				tempEnvironment = 0.5 * (tempEnvironmentNode[iNprev] + tempEnvironmentNode[iN]);
				cpNode = cpNodePrev;
				tempkm1 = tempNodePrev;
				ebkm1 = CalcEnergyBalance(qMass, tempkm1, tempNodePrev,
						tempEnvironment, dL, cpNode, cpNodePrev, outerDiamSeg);
				tempk = tempNodePrev + 0.1; // just a first estimate
			} else // TemperatureCalculationOption.Explicit
			{
				tempk = temperatureNode[iN];
				tempkm1 = tempk;
				tempEnvironment = 0;
				ebkm1 = 0;
			}

			boolean pbConverged = false; // pressure balance converged?
			boolean ebConverged = false; // energy balance converged?
			boolean converged = false; // N-R converged?
			int k = 0; // N-R iteration number
			while (!converged) {
				k++;

				// cut off iteration
				if (k > 100) {
					notConverting = true;
					break;
				}

				// Newton Raphson is applied sequentially on pressure and
				// temperature balance
				// with pressure and temperature as variables, respectively.

				preskm2 = preskm1;
				preskm1 = presk;
				tempkm2 = tempkm1;
				tempkm1 = tempk;

				// calculate pressure update from pressure balance

				if (tempCalcOption == PipeTempCalcMethod.Implicit) {
					// pbkm2 has to be recalculated based on new temperature
					densAvg = 0.5 * (densNodePrev + TnoWaterProperties.Density(
							preskm2, tempkm1, salinityNode[iN]));
					viscAvg = 0.5 * (viscNodePrev + TnoWaterProperties
							.Viscosity(preskm2, tempkm1, salinityNode[iN]));
					pbkm2 = CalcPressureBalance(preskm2, presNodePrev, qMass,
							dZ, dL, densAvg, viscAvg, iSegment, signFlow
									* signCalc);
				} else // TemperatureCalculationOption.Explicit
				{
					// previous pbkm1 can be used as temperature is constant
					pbkm2 = pbkm1;
				}

				densAvg = 0.5 * (densNodePrev + TnoWaterProperties.Density(
						preskm1, tempkm1, salinityNode[iN]));
				viscAvg = 0.5 * (viscNodePrev + TnoWaterProperties.Viscosity(
						preskm1, tempkm1, salinityNode[iN]));
				pbkm1 = CalcPressureBalance(preskm1, presNodePrev, qMass, dZ,
						dL, densAvg, viscAvg, iSegment, signFlow * signCalc);

				dPBdp = (pbkm2 - pbkm1) / (preskm2 - preskm1);
				presk = preskm1 - pbkm1 / dPBdp;

				pbConverged = Math.abs(pbkm1) < pbTolerance;

				// calculate temperature update from energy balance

				if (tempCalcOption == PipeTempCalcMethod.Implicit) {
					ebkm2 = ebkm1; // energy balance is independent of pressure
									// as cp is independent on pressure, hence
									// previous eb can be used
					cpNode = TnoWaterProperties.HeatCapacity(preskm1, tempkm1,
							salinityNode[iN]);
					ebkm1 = CalcEnergyBalance(qMass, tempkm1, tempNodePrev,
							tempEnvironment, dL, cpNode, cpNodePrev, outerDiamSeg);
					dEBdT = (ebkm2 - ebkm1) / (tempkm2 - tempkm1);
					tempk = tempkm1 - ebkm1 / dEBdT;
					ebConverged = Math.abs(ebkm1) < ebTolerance;
				} else // TemperatureCalculationOption.Explicit
				{
					ebConverged = true;
				}

				converged = pbConverged && ebConverged;
			}

			// update Node properties
			pressureNode[iN] = presk;
			temperatureNode[iN] = tempk;
			densityNode[iN] = TnoWaterProperties.Density(pressureNode[iN],
					temperatureNode[iN], salinityNode[iN]);
			viscosityNode[iN] = TnoWaterProperties.Viscosity(pressureNode[iN],
					temperatureNode[iN], salinityNode[iN]);
			heatCapacityNode[iN] = TnoWaterProperties.HeatCapacity(
					pressureNode[iN], temperatureNode[iN], salinityNode[iN]);
			qVolNode[iN] = Qmass / densityNode[iN];
		}

		// calculate and store total results
		DpTotal = pressureNode[NodesCount - 1] - pressureNode[0];
		DTTotal = temperatureNode[NodesCount - 1] - temperatureNode[0];
		DpGravTotal = 0;
		DpViscTotal = 0;
		DpPumpTotal = 0;
		QvolAvg = qVolNode[0];
		for (int iN = 1; iN < NodesCount; iN++) {
			int iSegment = iN - 1;
			dL = lNode[iN] - lNode[iN - 1];
			dZ = zNode[iN] - zNode[iN - 1];
			densAvg = 0.5 * (densityNode[iN] + densityNode[iN - 1]);
			viscAvg = 0.5 * (viscosityNode[iN] + viscosityNode[iN - 1]);

			dPgravSegment[iSegment] = DpGravity(dZ, densAvg);
			dPviscSegment[iSegment] = DpViscous(qMass, dL, densAvg, viscAvg,iSegment);
			dPpumpSegment[iSegment] = DpPump(iSegment, signFlow);
			DpGravTotal += dPgravSegment[iSegment];
			DpViscTotal += dPviscSegment[iSegment];
			DpPumpTotal += dPpumpSegment[iSegment];
			QvolAvg += qVolNode[0];
		}
		QvolAvg /= NodesCount;

	}

	private double CalcPressureBalance(double presNode, double presNodePrev,
			double qMass, double dZ, double dL, double densAvg, double viscAvg,
			int iSegment, double signPump) {
		double dpGrav = DpGravity(dZ, densAvg);
		double dpVisc = DpViscous(qMass, dL, densAvg, viscAvg,iSegment);
		double dpPump = DpPump(iSegment, signPump);
		double pressureBalance = presNode - presNodePrev - dpGrav - dpVisc
				- dpPump;
		return pressureBalance;
	}

	private double CalcEnergyBalance(double qMass, double tempNode,
			double tempNodePrev, double tempEnvironment, double dL,
			double cpNode, double cpNodePrev, double outerDiameterSegment) {

		// these parameters are fixed for time being
		double kt = 3; // thermisch conductiviteit van het gesteente (W/(m.K))
		double at = 1.2e-6; // thermisch diffusiteit van het gesteente (m2/s)
		double time = 365 * 24 * 3600; // one year

		// heat loss per length (W/m); heat loss to environment is +
		double qheat = 4
				* Math.PI
				* kt
				* ((tempNode + tempNodePrev) * 0.5 - tempEnvironment)
				/ Math
						.log(16
								* at
								* time
								/ (TnoUnits.ExpEulerConstant * outerDiameterSegment * outerDiameterSegment));

		// Note, as sign(qMass) indicates flow direction and sign(dL)
		// calculation direction signs are OK
		double energyBalance = qMass
				* (tempNode * cpNode - tempNodePrev * cpNodePrev) / dL + qheat; // W/m

		return energyBalance;
	}

	private double DpGravity(double dZ, double density) {
		double dP = -TnoUnits.GravConstant * dZ * density; // dP negative for
															// positive dZ
		return dP;
	}

	private double DpViscous(double Qmass, double dL, double density,
			double viscosity, int iSegment) {
		double dP;

		if (Qmass == 0) {
			dP = 0.0;
		} else {
			double Qvol = Math.abs(Qmass / density);
			double v = Qvol / (Math.PI * .25 * InnerDiameterSegment[iSegment] * InnerDiameterSegment[iSegment]);

			double Re = density * v * InnerDiameterSegment[iSegment] / viscosity;

			double f;
			// f = 64.0 / Re; // laminair
			//double relRoughness = RoughnessSegment[iSegment] / InnerDiameterSegment[iSegment];

			double tempVariable = (1.14 - 2 * Math.log10(RoughnessSegment[iSegment]
					/ InnerDiameterSegment[iSegment] + 21.25 / Math.pow(Re, 0.9)));
			f = 1 / (tempVariable * tempVariable);

			double dPsign = Qmass > 0 ? -1 : +1; // flow in positive direction
													// gives pressure drop
			dP = dPsign * f * density * dL * v * v / (2 * InnerDiameterSegment[iSegment]);
		}
		return dP;
	}

	private double DpPump(int iSegment, double signDpPump) {
		double dP = signDpPump * (iSegment == PumpSegment ? PumpPressure : 0);
		return dP;
	}

	public void SetPump(double pumpZSpec) {
                // for locating the pump
		Zbegin = 0;
                Zend = - SectionDepthTVD[SectionDepthTVD.length-1];

		PumpPresent = true;
		PumpZ = pumpZSpec; // set field behind property PumpZ

		if (Zbegin > Zend)
			if (PumpZ >= Zbegin) // pump above begin
			{
				PumpSegment = 0;
			} else if (PumpZ <= Zend) // pump below end
			{
				PumpSegment = SegmentsCount - 1;
			} else {
				for (int iN = 1; iN < NodesCount; iN++) {
					if (PumpZ > zNode[iN]) {
						PumpSegment = iN - 1;
						break;
					}
				}
			}
		else if (Zbegin < Zend)
			if (PumpZ >= Zend) // pump below end
			{
				PumpSegment = SegmentsCount - 1;
			} else if (PumpZ <= Zbegin) // pump above begin
			{
				PumpSegment = 0;
			} else {
				for (int iN = 1; iN < NodesCount; iN++) {
					if (PumpZ < zNode[iN]) {
						PumpSegment = iN - 1;
						break;
					}
				}
			}
		else {
			// System.out.println("Warning pipe " + Name +
			// ". Pump specified but Zbegin = Zend. Pump set in segment 0.");
			PumpSegment = 0;
		}

	}

	public void RemovePump() {
		PumpZ = 0; // set field behind property PumpZ
		PumpPressure = 0;
		PumpMaxRate = 0;
		PumpEfficiency = 0;

		PumpPresent = false;
		PumpSegment = 0;
	}

        private double[] cloneDoubleArray(double[] inputList){

		int length = inputList.length;
		double[] outputList = new double[length];
		for (int i=0; i<length; i++){
			outputList[i] = inputList[i];
		}
		return outputList;
	}

	public TnoPipe Clone() {
            TnoPipe c = new TnoPipe();
            c.lNode = cloneDoubleArray(this.lNode); // Length Along Pipe of Node
            c.zNode = cloneDoubleArray(this.zNode); // z-coordinate of Node
            c.pressureNode = cloneDoubleArray(this.pressureNode); // (Pa)
            c.temperatureNode = cloneDoubleArray(this.temperatureNode); // (degC)
            c.tempEnvironmentNode = cloneDoubleArray(this.tempEnvironmentNode); // (degC)
            c.salinityNode = cloneDoubleArray(this.salinityNode); // (1)
            c.densityNode = cloneDoubleArray(this.densityNode); // (kg/m3)
            c.viscosityNode = cloneDoubleArray(this.viscosityNode); // (Pa.s)
            c.heatCapacityNode = cloneDoubleArray(this.heatCapacityNode); // (J/(kg.K))
            c.qVolNode = cloneDoubleArray(this.qVolNode); // (m3/s)
            c.dPgravSegment = cloneDoubleArray(this.dPgravSegment); // (Pa)
            c.dPviscSegment = cloneDoubleArray(this.dPviscSegment); // (Pa)
            c.dPpumpSegment = cloneDoubleArray(this.dPpumpSegment); // (Pa)
            c.notConverting = this.notConverting;
            c.Name = this.Name;
            c.Length = this.Length; // (m)
            c.InnerDiameter = cloneDoubleArray(this.InnerDiameter); // (m)
            c.OuterDiameter = this.OuterDiameter; // (m)
            c.Roughness = cloneDoubleArray(this.Roughness); // (m)
            c.Zbegin = this.Zbegin; // (m)
            c.Zend = this.Zend; // (m)
            c.SectionDepthAH = cloneDoubleArray(this.SectionDepthAH);
            c.SectionDepthTVD = cloneDoubleArray(this.SectionDepthTVD);
            c.SectionAngle = cloneDoubleArray(this.SectionAngle);
            c.InnerDiameterSegment = cloneDoubleArray(this.InnerDiameterSegment); //(m)
            c.RoughnessSegment = cloneDoubleArray(this.RoughnessSegment); //(m)
            c.SegmentAngle = cloneDoubleArray(this.SegmentAngle);
            c.SegmentIndex = cloneDoubleArray(this.SegmentIndex);
            c.PumpZ = this.PumpZ;
            c.PumpPressure = this.PumpPressure; // (Pa)
            c.PumpMaxRate = this.PumpMaxRate; // (m3/s)
            c.PumpEfficiency = this.PumpEfficiency; // (1)
            c.TemperatureCalculationOption = this.TemperatureCalculationOption;
            c.Qmass = this.Qmass; // (kg/s)
            c.QvolAvg = this.QvolAvg;
            c.DpTotal = this.DpTotal;
            c.DTTotal = this.DTTotal;
            c.DpGravTotal = this.DpGravTotal;
            c.DpViscTotal = this.DpViscTotal;
            c.DpPumpTotal = this.DpPumpTotal;
            c.SegmentsCount = this.SegmentsCount;
            c.SegmentLength = this.SegmentLength; // (m)
            c.NodesCount = this.NodesCount;
            c.PumpSegment = this.PumpSegment; // number of Segment with Pump
            c.PumpPresent = this.PumpPresent;

            return c;
        }
}
