package tno.geoenergy.doubletcalc.io.file;

/**
 * @author Kronimus
 * 
 * 
 *          (c) 2009 TNO
 *          
 *   Disclaimer
 *	 see
 *   http://www.tno.nl/downloads/Disclaimer%20websites%20TNO.UK1.pdf
 *	 and
 *	 http://www.nlog.nl/nl/home/termsNLOG.html          
 * 
 */

import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;
import tno.geoenergy.doubletcalc.ResultSet;
import tno.geoenergy.doubletcalc.TnoPipe;
import tno.geoenergy.doubletcalc.TnoUnits;
import tno.geoenergy.doubletcalc.stochastic.TnoMyStochasticObjects;
import tno.geoenergy.doubletcalc.stochastic.TnoSingularContainer;
import tno.geoenergy.doubletcalc.stochastic.TnoStochasticContainerVar;

public class Dataexport {

	private Dataexport() {
	};

	private static double calcAverage(double[] nodeList) {

		double i = 0;
		double sum = 0;
		for (double node : nodeList) {
			sum += node;
			i++;
		}

		return sum / i;
	}

	private static double calcDiff(double[] nodeList) {

		double first = (float) nodeList[0];
		int maxIndex = nodeList.length - 1;
		double last = (float) nodeList[maxIndex];

		return last - first;

	}

	public static void createBaseCaseCSV(File filename, ResultSet rs) {

		final TnoPipe PRODUCER = rs.baseCaseProducerPipe;
		final TnoPipe INJECTOR = rs.baseCaseInjectorPipe;
                final TnoPipe HYDROSTATICPRODUCER = rs.hydrostaticProducer;
		final TnoPipe HYDROSTATICINJECTOR = rs.hydrostaticInjector;
		final String SEP = ", "; // separator;

		try {
			BufferedWriter bw = new BufferedWriter(new FileWriter(filename));

			// master header
			bw.write("Doublet Calculator 1.4 Base Case Details");
			bw.newLine();
			bw.newLine();

                        TnoPipe[] aquifers = { HYDROSTATICPRODUCER, HYDROSTATICINJECTOR};
			for (TnoPipe pipe : aquifers) {

				String pipeID;
				if(pipe.equals(HYDROSTATICPRODUCER))
					pipeID = "HYDROSTATIC AQUIFER PROPERTIES @PRODUCER";
                                else if(pipe.equals(HYDROSTATICINJECTOR))
					pipeID = "HYDROSTATIC AQUIFER PROPERTIES @INJECTOR";
				else
					pipeID = pipe.Name;

				// pipe header
				bw.write("=== " + pipeID + " ===");
				bw.newLine();
				bw.write("Z(m), P(bar), T(degC), S(ppm), Density(kg/m3), Viscosity(Pa s)");
				bw.newLine();

				// all rows
				int maxInd = pipe.lNode.length;
                                
				for (int n = 0; n < maxInd; n++) {

					bw.write( pipe.zNode[n] + SEP
                                                + pipe.pressureNode[n] / 1e5 + SEP
                                                + pipe.temperatureNode[n] + SEP
                                                + (int) (pipe.salinityNode[n] * 1e6) + SEP
                                                + pipe.densityNode[n] + SEP 
                                                + pipe.viscosityNode[n]);
					bw.newLine();
				}
				bw.newLine();
				bw.newLine();
			}

			TnoPipe[] pipes = {PRODUCER, INJECTOR };
			for (TnoPipe pipe : pipes) {

				String pipeID;
				if (pipe.equals(PRODUCER))
					pipeID = "PRODUCER";
                                else if(pipe.equals(INJECTOR))
					pipeID = "INJECTOR";
				else
					pipeID = pipe.Name;

				// pipe header
				bw.write("=== " + pipeID + " ===");
				bw.newLine();
				bw
						.write("iN, Segment, L(m), Z(m), Angle (deg), Inner diameter(inch), Roughness(milli-inch), P(bar), T(degC), S(ppm), Density(kg/m3), Viscosity(Pa s), Qvol(m3/h), dPGrav(bar), dPVisc(bar), dPpump(bar)");
				bw.newLine();
				
				// first row
				bw.write(0 + SEP
						+ "" + SEP
						+ pipe.lNode[0] + SEP
						+ pipe.zNode[0] + SEP
						+ "" + SEP
						+ "" + SEP
						+ "" + SEP
						+ pipe.pressureNode[0] / 1e5 + SEP
						+ pipe.temperatureNode[0] + SEP
						+ (int) (pipe.salinityNode[0] * 1e6) + SEP
						+ pipe.densityNode[0] + SEP + pipe.viscosityNode[0]
						+ SEP + pipe.qVolNode[0] * 3600);
				bw.newLine();

				// following rows
				int maxInd = pipe.lNode.length; 

				for (int n = 1; n < maxInd; n++) {

					bw.write(n + SEP 
							+ pipe.SegmentIndex[n-1]+ SEP
							+ pipe.lNode[n] + SEP + 
							+ pipe.zNode[n]
							+ SEP + Math.toDegrees(pipe.SegmentAngle[n-1])
							+ SEP + pipe.InnerDiameterSegment[n-1]*TnoUnits.SI_Inch + SEP
							+ pipe.RoughnessSegment[n-1]*TnoUnits.SI_Inch*1000
							+ SEP + pipe.pressureNode[n] / 1e5 + SEP
							+ pipe.temperatureNode[n] + SEP
							+ (int) (pipe.salinityNode[n] * 1e6) + SEP
							+ pipe.densityNode[n] + SEP + pipe.viscosityNode[n]
							+ SEP + pipe.qVolNode[n] * 3600 + SEP
							+ pipe.dPgravSegment[n - 1] / 1e5 + SEP
							+ pipe.dPviscSegment[n - 1] / 1e5 + SEP
							+ pipe.dPpumpSegment[n - 1] / 1e5);
					bw.newLine();
				}

				// last row
				bw.write("total/average/difference" + SEP
						+ pipe.SegmentIndex[pipe.SegmentIndex.length-1] + SEP
						+ calcDiff(pipe.lNode) 
						+ SEP + calcDiff(pipe.zNode)
						+ SEP + Math.toDegrees(calcAverage(pipe.SegmentAngle))
						+ SEP + calcAverage(pipe.InnerDiameterSegment)*TnoUnits.SI_Inch
						+ SEP + calcAverage(pipe.RoughnessSegment)*TnoUnits.SI_Inch*1000
						+ SEP + pipe.DpTotal / 1e5 + SEP + pipe.DTTotal + SEP
						+ (int) (calcAverage(pipe.salinityNode) * 1e6) + SEP
						+ calcAverage(pipe.densityNode) + SEP
						+ calcAverage(pipe.viscosityNode) + SEP
						+ calcAverage(pipe.qVolNode) * 3600 + SEP
						+ pipe.DpGravTotal / 1e5 + SEP + pipe.DpViscTotal / 1e5
						+ SEP + pipe.DpPumpTotal / 1e5);

				bw.newLine();
				bw.newLine();
				bw.newLine();
			}


			
			// doublet nodes
			bw.write("=== DOUBLET NODES ===");
			bw.newLine();
			bw.write("Node, , P(bar), T(degC)");
			
			int maxIndex = rs.doubletNodes.length-1; //skip Residue node
			
			for (int i=0; i < maxIndex; i++){
				bw.newLine();
				bw.write(rs.doubletNodes[i] + ", " + (rs.baseCasePressureAlongDoublet[i] * 1e-5) + ", " +
						rs.baseCaseTemperatureAlongDoublet[i]);
			}
			bw.newLine();
			bw.newLine();
			bw.newLine();

			// base case results
			bw.write("=== BASE CASE RESULTS ===");
			bw.newLine();

                        for (String identifier : ResultSet.getBaseGeoOutputProperties()){
                            bw.write(TnoSingularContainer.find(identifier).getTitle() + ", "
					+ TnoSingularContainer.find(identifier).getValueForm());
                            bw.newLine();
                        }

                        bw.newLine();

                        for (String identifier : ResultSet.getBaseCaseWellProperties()){
                            bw.write(TnoSingularContainer.find(identifier).getTitle() + ", "
					+ TnoSingularContainer.find(identifier).getValueForm());
                            bw.newLine();
                        }
			
                        bw.newLine();
                        bw.newLine();

                        bw.write("=== STOCHASTIC RESULTS ===");
                        bw.newLine();
                        bw.write(", P90, P50, P10");
                        bw.newLine();

                        for (String identifier : ResultSet.getStochGeoOutputProperties()){
                            bw.write(TnoStochasticContainerVar.find(identifier).getTitle() +
					", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(90.0)+
                                        ", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(50.0)+
                                        ", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(10.0));
                            bw.newLine();
                        }

                        bw.newLine();

                        for (String identifier : ResultSet.getStochWellProperties()){
                            bw.write(TnoStochasticContainerVar.find(identifier).getTitle() +
					", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(90.0)+
                                        ", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(50.0)+
                                        ", " + TnoStochasticContainerVar.find(identifier).calcPValueForm(10.0));
                            bw.newLine();
                        }

                        bw.newLine();
			bw.write("=== end of file ===");

			bw.flush();
			bw.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			StringToFile.showNotFoundError();
		} catch (IOException e) {
			e.printStackTrace();
			StringToFile.showIOError();
		}
	}

}; // class

