package tno.geoenergy.doubletcalc.stochastic;

/**
 * @author Van Wees
 * 
 * 
 *          (c) 2009 TNO
 *          
 *   Disclaimer
 *	 see
 *   http://www.tno.nl/downloads/Disclaimer%20websites%20TNO.UK1.pdf
 *	 and
 *	 http://www.nlog.nl/nl/home/termsNLOG.html          
 * 
 */

import java.util.Random;

public class TnoDistributionVarTriangular extends TnoDistributionVariable {

	public static double SD_PVALUE = 0.1587;
	
	public TnoDistributionVarTriangular(double min, double median, double max) {
		super(min, median, max);
	}

	public TnoDistributionVarTriangular(double min, double median, double max,
			int seed) {
		super(min, median, max);
	}

	public void setParameters(double min, double median, double max, String name) {
		if (min >= median || median >= max) {

		} else {
			a = min;
			b = median;
			c = max;

			mean = 0;
			// left triangle
			if (a == b) {
				mean += 0.5 * b;
			} else {
				mean += 1 / (6 * (b - a) * (b - a))
						* (2 * b * b * b - 3 * a * b * b + a * a * a);
			}

			// right triangle
			if (b == c) {
				mean += 0.5 * b;
			} else {
				mean += 1 / (6 * (c - b) * (c - b))
						* (2 * b * b * b - 3 * c * b * b + c * c * c);
			}

		}
	}

	public void setParameters(double min, double median, double max) {
		setParameters(min, median, max, null);
	}

	public void setParameters(double min, double median, double max,
			String name, int seed) {
		randomNumber = new Random(seed);
		setParameters(min, median, max, name);
	}

	public void SetParameters(double min, double median, double max, int seed) {
		setParameters(min, median, max, null, seed);
	}

	public double PDF(double x) {
		double pdf;
		if (x < a)
			pdf = 0;
		else if (x < b)
			pdf = (x - a) / ((b - a) * (b - a));
		else if (x == b)
			pdf = 0.5 * ((x - a) / ((b - a) * (b - a)) + (c - x)
					/ ((c - b) * (c - b)));
		else if (x < c)
			pdf = (c - x) / ((c - b) * (c - b));
		else
			pdf = 0;

		return pdf;
	}

	public double CDF(double x) {
		double cdf;
		if (x < a)
			cdf = 0;
		else if (x < b)
			cdf = 0.5 * (x - a) * (x - a) / ((b - a) * (b - a));
		else if (x == b)
			cdf = 0.5;
		else if (x < c)
			cdf = 1 - 0.5 * (c - x) * (c - x) / ((c - b) * (c - b));
		else
			cdf = 1;

		return cdf;
	}

	public double NextValue() {
		double s = randomNumber.nextDouble(); // get sample= cdf value
		if (s < 0.5)
			return a + (b - a) * Math.sqrt(2 * s);
		else if (s == 0.5)
			return b;
		else
			return c - (c - b) * Math.sqrt(2 * (1 - s));
	}

	public void constructACFromP10P90(double p10, double p90, boolean minzero)
	  {
		  a = (p90 - b*Math.sqrt(0.2))/ (1-Math.sqrt(0.2));
		  if (minzero) a = Math.max(0,a); 
		  c = (p10 - b*Math.sqrt(0.2))/ (1-Math.sqrt(0.2));
	  }
	  
	public void constructACFromPSD(double pluspsd, double minpsd, boolean minzero) {
		// TODO Auto-generated method stub
		a = (minpsd - b*Math.sqrt(2*SD_PVALUE))/ (1-Math.sqrt(2*SD_PVALUE));
		if (minzero) a = Math.max(0,a); 
		c = (pluspsd- b*Math.sqrt(2*SD_PVALUE))/ (1-Math.sqrt(2*SD_PVALUE));


		System.out.println("STDEV (a2+b2+c2 -ab-ac-bc)/18 :" + Math.sqrt((a*a+b*b+c*c-a*b-a*c-b*c)/18));
	}
	
	@Override
	public double[] generateSamples(int numSamples) {
		double[] samples = new double[numSamples];
		int r = 0;
		int[] hussle = new int[numSamples];
		hussle = createRandomIndexArray(numSamples);
		double sample;

		for (int comp = 0; comp < numSamples; comp++) {
			sample = (comp + randomNumber.nextDouble()) / (numSamples * 1.0);
			samples[hussle[r++]] = getPDF_Value(sample);
		}

		return samples;
	}

	private double getPDF_Value(double s) {

		if (s < 0.5) {
			if ((b - a) < 1e-20) {
				return a;
			}
			return a + (b - a) * Math.sqrt(2 * s);
		} else if (s == 0.5) {
			return b;
		} else {
			if ((c - b) < 1e-20) {
				return b;
			}
			return c - (c - b) * Math.sqrt(2 * (1 - s));
		}

	}

	private int[] createRandomIndexArray(int numSamples) {

		int index[] = new int[numSamples];

		for (int i = 0; i < numSamples; i++) {
			index[i] = i;
		}

		for (int i = 0; i < numSamples; i++) {
			int randomIndex1 = i;
			int randomIndex2 = (int) Math.floor(randomNumber.nextDouble()
					* numSamples);
			if (randomIndex1 < 0)
				randomIndex1 = 0;
			if (randomIndex2 > numSamples)
				randomIndex2 = numSamples;
			int swapValue = index[randomIndex1];
			index[randomIndex1] = index[randomIndex2];
			index[randomIndex2] = swapValue;
		}
		for (int i = 0; i < numSamples; i++) {
			int randomIndex1 = numSamples - i - 1;
			int randomIndex2 = (int) Math.floor(randomNumber.nextDouble()
					* numSamples);
			if (randomIndex1 < 0)
				randomIndex1 = 0;
			if (randomIndex2 > numSamples)
				randomIndex2 = numSamples;
			int swapValue = index[randomIndex1];
			index[randomIndex1] = index[randomIndex2];
			index[randomIndex2] = swapValue;
		}

		return index;
	}

}
