package tno.geoenergy.doubletcalc.stochastic;

/**
 * @author Van Wees
 * 
 * 
 *         (c) 2009 TNO
 *         
 *   Disclaimer
 *	 see
 *   http://www.tno.nl/downloads/Disclaimer%20websites%20TNO.UK1.pdf
 *	 and
 *	 http://www.nlog.nl/nl/home/termsNLOG.html         
 * 
 */

public class TnoStochasticVariable extends TnoStochasticObject {

	protected double[] realisations = null;

	public double[] getRealisations() {
		return realisations;
	}

	public float[] getSortedRealisationsAsFloat() {
		double[] sortedRealisations = sort();
		float[] v = new float[this.numSamples];
		for (int i = 0; i < numSamples; i++) {
			v[i] = (float) sortedRealisations[i];
		}
		return v;
	}

	public void setRealisations(double[] realisations) {
		this.realisations = realisations;
	}

	protected int numSamples = 0;
	public TnoDistributionVariable distribution = null;

	public TnoDistributionVariable getDistribution() {
		return distribution;
	}

	public void setDistribution(TnoDistributionVariable distribution) {
		this.distribution = distribution;
	}

	/**
	 * 
	 * @param sampleArray
	 * 
	 *            this routine generates a realisations based on set of samples,
	 *            from elsewhere (e.g. typically for a yearly slice in a
	 *            timeseries)
	 * 
	 *            this is for temporary services and not added to global list
	 * 
	 */

	public TnoStochasticVariable(double[] samples) {
		this.realisations = samples;
		numSamples = samples.length;
	}

	public double NextValue(int i) {
		if (i == 0) {
			this.buildRealisations();
		}
		return this.realisations[i];

	}

	/**
	 * 
	 * @param nSamples
	 * 
	 *            this routine generates an (output) realisations object,
	 *            without an attached distribution
	 * 
	 */

	public TnoStochasticVariable(int nSamples) {
		numSamples = nSamples;
		this.createEmptyRealisations();
	}

	/**
	 * 
	 * @param nSamples
	 * @param dist
	 * 
	 *            this constructor generates a realisations object, related to
	 *            an input distribution
	 * 
	 */

	public TnoStochasticVariable(int nSamples, TnoDistributionVariable dist) {
		numSamples = nSamples;
		distribution = dist;
	}

	public TnoStochasticVariable(int nSamples, TnoDistributionVarTriangular dist) {
		numSamples = nSamples;
		distribution = dist;
	}

	public void setSamplesAndDist(int nSamples,
			TnoDistributionVarTriangular dist) {
		this.numSamples = nSamples;
		this.distribution = dist;
	}

	public TnoStochasticVariable(String name, String units, int idType) {
		super(name, units, idType);
		numSamples = TnoRunControl.numSamples;
		distribution = null;
	}

	public TnoStochasticVariable(String name, String units, int idType,
			double unitFactor, int round) {
		super(name, units, idType);
		numSamples = TnoRunControl.numSamples;
		distribution = null;
		this.unitFactor = unitFactor;
		this.round = round;
	}

	public void buildRealisations() {
		if (distribution != null) {
			realisations = distribution.generateSamples(numSamples);
		} else {
			this.createEmptyRealisations();
		}
	}

	public double size() {
		if (realisations != null) {
			return realisations.length;
		} else {
			return 0;
		}
	}

	/**
	 * 
	 * @param i
	 * @param v
	 * 
	 *            set realisation[i] to value v
	 * 
	 *            this is used for output generation
	 * 
	 */

	public void setSample(int i, double v) {
		realisations[i] = v;
	}

	/**
	 * Returns the max over the realisations.
	 * 
	 * @return double
	 */
	public final double calcMax() {
		double result = 0;
		for (int i = 0; i < size(); i++) {
			double value = realisations[i];
			if (i == 0) {
				result = value;
			} else if (result < value) {
				result = value;
			}
		}
		return result;
	}

	/**
	 * Returns the mean over the realisations.
	 * 
	 * @return double
	 */
	public final double calcMean() {
		double total = 0;
		for (int i = 0; i < size(); i++) {
			total += realisations[i];
		}

		return total / size();
	}

	/**
	 * Returns the min over the realisations.
	 * 
	 * @return double
	 */
	public final double calcMin() {
		double result = 0;
		for (int i = 0; i < size(); i++) {
			double value = realisations[i];
			if (i == 0) {
				result = value;
			} else if (result > value) {
				result = value;
			}
		}
		return result;
	}

	/**
	 * Returns the expected downside over the realisations.
	 * 
	 * @return double
	 */
	public final double calcDownside() {
		double result = 0;
		for (int i = 0; i < size(); i++) {
			double value = realisations[i];
			if (value < 0) {
				result += value;
			}
		}
		return result / size();
	}

	/**
	 * Returns the expected upside
	 * 
	 * @return double
	 */
	public final double calcUpside() {
		double result = 0;
		for (int i = 0; i < size(); i++) {
			double value = realisations[i];
			if (value > 0) {
				result += value;
			}
		}
		return result / size();
	}

	/**
	 * Calculates the value for P10 (or P15, P50 etc ) by just returning the
	 * value that is bigger than 10% (or 15% or 50% ) of the realisations.
	 * 
	 * @return double
	 * @param chance
	 *            double
	 */

	public final double calcPValue(double chance) {

		// Check for invalid input.
		if ((chance > 100) || (chance < 0)) {
			// logger.error( "Invalid chance given : P" + chance );
			throw new RuntimeException("Invalid chance given : P" + chance);
		}
		// Chance goes from 100 to 0 instead of from 0 - 100
		chance = 100 - chance;

		// If there is only one value just return the value.
		int sampleCount = realisations.length;
		if (sampleCount == 1) {
			return realisations[0];
		}

		// Sort the realisations.
		double[] sortedRealisations = sort();

		// Find the realisation that is on the border.
		int chosenRealisation = (int) ((double) sampleCount * chance / 100.0);

		// Check for strange results.
		if (chosenRealisation >= sampleCount) {
			return sortedRealisations[sampleCount - 1];
		}

		return sortedRealisations[chosenRealisation];
	}

	public final double calcPValueForm(double chance) {

		double pvalue = calcPValue(chance);

		return format(pvalue);

	}

	/**
	 * 
	 * returns P(sample >= x)
	 * 
	 * @param x
	 * @return double
	 */

	public double calcPvalueX(double x) {
		// Sort the realisations.
		double p = this.size();
		double[] sortedRealisations = sort();
		for (int i = 0; i < size(); i++) {
			if (sortedRealisations[i] >= x) {
				return (p * 1.0 / size());
			} else {
				p--;
			}
		}
		return (p * 1.0 / size());
	}

	/**
	 * Returns the standard deviation over the realisations.
	 * 
	 * @return double
	 */

	public final double calcStandardDeviation() {
		return Math.sqrt(calcVariance());
	}

	/**
	 * Returns the variance over the realisations.
	 * 
	 * @return double
	 */

	public final double calcVariance() {

		// variance = sum (X21..X2n) / n - mean2
		double total = 0;
		for (int i = 0; i < size(); i++) {
			total += (realisations[i] * realisations[i]);
		}

		if (total == 0) {
			return 0;
		}
		double mean = calcMean();
		return Math.abs((total / size()) - (mean * mean));

	}

	// JDJD additions for statistics

	/**
	 * Calculates the width of the range of the distribution
	 * 
	 * @return double
	 */
	public final double calcRangeWidth() {
		return (calcMax() - calcMin());
	}

	/**
	 * Calculates the Median e.g. see Spiegel, 1972, Shaums outline series of
	 * statistics, p. 47 median = p50
	 * 
	 * @return double
	 */
	public final double calcMedian() {
		return (calcPValue(50));
	}

	/**
	 * Calculates the Standard Error of the Mean e.g. see Spiegel, 1972, Shaums
	 * outline series of statistics, p. 144
	 * 
	 * standard error = sd/N
	 * 
	 * @return double
	 */
	public final double calcMeanStandardError() {

		return (calcStandardDeviation() / Math.sqrt(size()));
	}

	/**
	 * Calculates the Coefficient of Variability e.g. see Spiegel, 1972, Shaums
	 * outline series of statistics, p. 73
	 * 
	 * coefficient of variability = sd/mean
	 * 
	 * @return double
	 */
	public final double calcCoeffientOfVariability() {

		return (calcStandardDeviation() / calcMean());
	}

	/**
	 * Calculates the ith moment e.g. see Spiegel, 1972, Shaums outline series
	 * of statistics, p. 89
	 * 
	 * moment m = sum (X-mean)^m / n
	 * 
	 * second moment corresponds to variance
	 * 
	 * @return double
	 */
	public final double calcMoment(int m) {

		double mean = calcMean();
		double total = 0;
		for (int i = 0; i < size(); i++) {
			total += Math.pow(realisations[i] - mean, m);
		}

		if (total == 0) {
			return 0;
		}

		return (total / size());

	}

	/**
	 * Calculates the (moment coefficient of) skewness e.g. see Spiegel, 1972,
	 * Shaums outline series of statistics, p. 91
	 * 
	 * skewness = Moment3 / sd^3
	 * 
	 * @return double
	 */
	public final double calcSkewness() {

		double sd = calcStandardDeviation();
		double m3 = calcMoment(3);

		return (m3 / Math.pow(sd, 3));
	}

	/**
	 * Calculates the (moment coefficient of) kurtosis e.g. see Spiegel, 1972,
	 * Shaums outline series of statistics, p. 91
	 * 
	 * kurtosis = Moment4 / (Moment2*Moment2)
	 * 
	 * @return double
	 */
	public final double calcKurtosis() {

		double m2 = calcMoment(2);
		double m4 = calcMoment(4);

		return (m4 / (m2 * m2));
	}

	/**
	 * 
	 * create an empty set of realisations
	 * 
	 * @param sampleCount
	 *            number of realisations
	 */
	public void createEmptyRealisations() {

		if ((realisations == null || numSamples != realisations.length)) {
			realisations = new double[numSamples];
		}

		for (int i = 0; i < realisations.length; i++) {
			realisations[i] = 0;
		}

	}

	protected double[] sort() {
		return sort(realisations);
	}

	/**
	 * This method sorts an array of doubles from low to high using quick sort
	 * routine.
	 * 
	 * @return double[]
	 * @param original
	 *            double[]
	 * @exception java.lang.Exception
	 *                if sort fails
	 */

	protected double[] sort(double original[]) {

		if (original.length > 1) {
			double sorted[] = new double[original.length];
			System.arraycopy(original, 0, sorted, 0, original.length);
			java.util.Arrays.sort(sorted);
			return sorted;
		} else {
			return original;
		}

	}

}
